<?php

/**
*
* @package phpBB3 SEO Sitemap
* @copyright (c) 2014 www.phpbb-work.ru
* @license http://opensource.org/licenses/gpl-license.php GNU Public License
*
*/

namespace shredder\sitemap;

use Symfony\Component\DependencyInjection\ContainerInterface;

class core
{
	/**
	* Constructor
	* NOTE: The parameters of this method must match in order and type with
	* the dependencies defined in the services.yml file for this service.
	*

	/** @var \phpbb\auth\auth */
	private $auth;

	/** @var \phpbb\config\config */
	private $config;

	/** @var \phpbb\config\db_text */
	protected $config_text;

	/** @var \phpbb\db\driver\driver_interface */
	private $db;

	/** @var ContainerInterface */
	private $phpbb_container;

	/** @var \phpbb\extension\manager */
	private $phpbb_extension_manager;

	/** @var \phpbb\user */
	private $user;

	/** @var string */
	private $phpbb_root_path;
	private $php_ext;

	public function __construct(\phpbb\auth\auth $auth, \phpbb\config\config $config, \phpbb\config\db_text $config_text, \phpbb\db\driver\driver_interface $db, ContainerInterface $phpbb_container, \phpbb\extension\manager $phpbb_extension_manager, \phpbb\user $user, $root_path, $php_ext)
	{
		$this->auth = $auth;
		$this->config = $config;
		$this->config_text = $config_text;
		$this->db = $db;
		$this->phpbb_container = $phpbb_container;
		$this->phpbb_extension_manager = $phpbb_extension_manager;
		$this->user = $user;
		$this->root_path = $root_path;
		$this->php_ext = $php_ext;
	}

	/**
	* Generate sitemap
	*/
	public function generate_sitemap()
	{
		global $f_xml, $urls;

		$excl = ($this->config_text->get('sitemap_seo_excluded')) ? explode(',', $this->config_text->get('sitemap_seo_excluded')) : array();
		$limit = $this->config['sitemap_seo_url_limit'];
		$path = generate_board_url() . '/';
		$dir = 'store/shredder/';

		// Try to create "sitemap store" directory if it does not exist
		if (!file_exists($this->root_path . $dir))
		{
			@mkdir($this->root_path . $dir, 0777);
			phpbb_chmod($this->root_path . $dir, CHMOD_READ | CHMOD_WRITE);
		}

		if (file_exists($this->root_path . $dir) && is_dir($this->root_path . $dir))
		{
			// Purge all files in "sitemap store" directory if they are old
			try
			{
				$iterator = new \DirectoryIterator($this->root_path . $dir);
			}
			catch (\Exception $e)
			{
				return;
			}

			foreach ($iterator as $fileInfo)
			{
				if ($fileInfo->isDot())
				{
					continue;
				}

				$file = $fileInfo->getFilename();

				if (strpos($file, 'htaccess') === false)
				{
					unlink($this->root_path . $dir . $file);
				}
			}
		}
		else
		{
			trigger_error(sprintf($this->user->lang['SEOMAP_CANT_WRITE'], $path . $dir));
		}

		// Write out .htaccess file
		$ht_file = '<Files *>'."\r\n";
		$ht_file .= "\t".'Order Allow,Deny'."\r\n";
		$ht_file .= "\t".'Allow from All'."\r\n";
		$ht_file .= '</Files>';

		if (!file_exists($this->root_path . $dir . '.htaccess'))
		{
			$written = true;

			if (!($fp = @fopen($this->root_path . $dir . '.htaccess', 'w')))
			{
				$written = false;
			}

			if (!(@fwrite($fp, $ht_file)))
			{
				$written = false;
			}

			@fclose($fp);

			if ($written)
			{
				phpbb_chmod($this->root_path . $dir . '.htaccess', CHMOD_READ);
			}
			else
			{
				trigger_error(sprintf($this->user->lang['SEOMAP_CANT_WRITE'], $path . $dir));
			}
		}

		// Check out other extensions for compatibility
		if ($this->phpbb_extension_manager->is_enabled('phpbbseo/usu'))
		{
			$seo_core = $this->phpbb_container->get('phpbbseo.usu.core');
		}

		$seo = (isset($seo_core)) ? $seo_core->seo_opt['url_rewrite'] : 0;

		if ($this->phpbb_extension_manager->is_enabled('shredder/seo_topic_url'))
		{
			$seo_topic_url = (!$this->config['stc_mode']) ? true : false;
		}

		// Build sitemap
		$f_beg = '<?xml version="1.0" encoding="UTF-8"?>'."\r\n";
		$f_beg .= '<?xml-stylesheet type="text/xsl" href="' . $path . 'ext/shredder/sitemap/styles/sitemap.xsl"?>'."\r\n";
		$f_beg .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">'."\r\n";
		$f_end = '</urlset>';

		// Simulate guest permissions so a guest user can create sitemap. Backup the original data first
		$backup = array(
			'user_data'	=> $this->user->data,
			'auth'		=> $this->auth
		);

		$sql = 'SELECT * FROM ' . USERS_TABLE . '
				WHERE user_id = ' . ANONYMOUS;
		$result = $this->db->sql_query($sql);

		while ($row = $this->db->sql_fetchrow($result))
		{
			$this->user->data = $row;
			$this->auth->acl($this->user->data);
		}
		$this->db->sql_freeresult($result);

		// Select basic info
		$sql = 'SELECT f.forum_id, f.forum_type, f.forum_name, f.forum_topics_per_page, COUNT(t.topic_id) AS forum_topics
			FROM ' . FORUMS_TABLE . ' f, ' . TOPICS_TABLE . ' t
			WHERE t.topic_visibility = ' . ITEM_APPROVED . '
				AND t.forum_id = f.forum_id
			GROUP BY f.forum_id';
		$result = $this->db->sql_query($sql);

		$f_ids = array(0);
		while ($row = $this->db->sql_fetchrow($result))
		{
			if (!in_array($row['forum_id'], $excl) && $this->auth->acl_get('f_list', $row['forum_id']) && $row['forum_type'] == FORUM_POST)
			{
				$f_ids[] = $row['forum_id'];
				$forums[$row['forum_id']] = $row;
			}
		}
		$this->db->sql_freeresult($result);

		// Generate forums list
		$index = 1;
		$glob = $ann = $times = array();

		$list_ary = $this->auth->acl_getf('f_list', true);
		$list_ary = array_unique(array_keys($list_ary));

		// Extract the backup to set the $this->user and $this->auth constants to what they were before
		$this->auth = $backup['auth'];
		$this->user->data = $backup['user_data'];

		unset($backup);

		// Get max global announcement time
		$sql = 'SELECT MAX(topic_last_post_time) AS max_glob_time
			FROM ' . TOPICS_TABLE . '
			WHERE topic_type = ' . POST_GLOBAL . '
				AND topic_visibility = ' . ITEM_APPROVED . '
				AND ' . $this->db->sql_in_set('forum_id', $list_ary);
		$result = $this->db->sql_query($sql);
		$glob = array((int) $this->db->sql_fetchfield('max_glob_time'));
		$this->db->sql_freeresult($result);

		// Bring forums to sitemap
		$sql = 'SELECT forum_id, topic_id, topic_type, topic_last_post_time
			FROM ' . TOPICS_TABLE . '
			WHERE topic_visibility = ' . ITEM_APPROVED . '
				AND ' . $this->db->sql_in_set('forum_id', $f_ids) . '
			ORDER BY forum_id,
			CASE
				WHEN (topic_type = ' . POST_ANNOUNCE . ' OR topic_type = ' . POST_GLOBAL . ') THEN 1
				' . ((defined('POST_ARTICLE')) ? 'WHEN topic_type = ' . POST_ARTICLE . ' THEN 2' : '') . '
				WHEN topic_type = ' . POST_STICKY . ' THEN 3
				ELSE 4
			END, topic_last_post_time DESC';
		$result = $this->db->sql_query($sql);

		$page = $topic = $all_f = 0;

		while ($data = $this->db->sql_fetchrow($result))
		{
			$t_type = (int) $data['topic_type'];

			if ($t_type == POST_ANNOUNCE || $t_type == POST_GLOBAL)
			{
				$ann[] = $data['topic_last_post_time'];
			}
			else
			{
				++$topic;
				$times[] = $data['topic_last_post_time'];
			}
			++$all_f;

			$per = ($forums[$data['forum_id']]['forum_topics_per_page']) ? $forums[$data['forum_id']]['forum_topics_per_page'] : $this->config['topics_per_page'];

			if ($topic == $per || $forums[$data['forum_id']]['forum_topics'] == $all_f)
			{
				$f_id = (int) $data['forum_id'];

				if ($seo)
				{
					if ($page == 0)
					{
						$seo_core->set_url($forums[$f_id]['forum_name'], $f_id, $seo_core->seo_static['forum']);
						$f_url = append_sid("{$this->root_path}viewforum.$this->php_ext", 'f=' . $f_id);
					}
					else
					{
						$f_url = append_sid("{$this->root_path}viewforum.$this->php_ext", 'f=' . $f_id . "&amp;start=" . ($page * $per));
					}

					$f_url = $seo_core->drop_sid($f_url);
				}
				else
				{
					if ($page == 0)
					{
						$f_url = $path . "viewforum.$this->php_ext?f=$f_id";
					}
					else
					{
						$f_url = $path . "viewforum.$this->php_ext?f=$f_id&amp;start=" . ($page * $per);
					}
				}

				if ($urls && ($urls % ($limit * $index)) == 0)
				{
					$f_xml = $f_beg . $f_xml . $f_end;

					$written = true;

					if (!($w = @fopen($this->root_path . $dir . $index . '.xml', 'w')))
					{
						$written = false;
					}

					if (!(@fwrite($w, $f_xml)))
					{
						$written = false;
					}

					@fclose($w);

					if (!$written)
					{
						trigger_error(sprintf($this->user->lang['SEOMAP_CANT_WRITE'], $path . $dir));
					}

					$f_xml = '';
					++$index;
				}

				$time = gmdate('Y-m-d\TH:i:s+00:00', (int) max(array_merge($glob, $ann, $times)));

				$this->add_to_record($f_url, $time, $this->config['sitemap_seo_freq_f'], $this->config['sitemap_seo_prior_f']);

				++$page;
				$times = array();
				$topic = 0;

				if ($forums[$f_id]['forum_topics'] == $all_f)
				{
					$page = $all_f = 0;
					$ann = array();
				}
			}
		}
		$this->db->sql_freeresult($result);

		// Generate topics list
		$seo_add = ($seo) ? ', t.topic_url' : '';

		$sql = 'SELECT t.topic_id' . $seo_add . ', t.topic_type, t.topic_title, t.topic_last_post_id, t.forum_id, p.post_id, p.post_modified
			FROM ' . TOPICS_TABLE . ' t, ' . POSTS_TABLE . ' p
			WHERE t.topic_visibility = ' . ITEM_APPROVED . ' AND p.post_visibility = ' . ITEM_APPROVED . '
				AND t.topic_status <> ' . ITEM_MOVED . '
				AND ' . $this->db->sql_in_set('t.forum_id', $f_ids) . '
				AND t.topic_id = p.topic_id
			ORDER BY t.topic_id, p.post_time';
		$result = $this->db->sql_query($sql);

		$times = array();
		$per = $this->config['posts_per_page'];
		$page = $post = 0;

		while ($data = $this->db->sql_fetchrow($result))
		{
			++$post;
			$times[] = $data['post_modified'];

			if ($post == $per || $data['post_id'] == $data['topic_last_post_id'])
			{
				$t_id = (int) $data['topic_id'];
				$f_id = (int) $data['forum_id'];
				$t_type = (int) $data['topic_type'];

				if ($seo)
				{
					if ($page == 0)
					{
						$seo_core->prepare_iurl($data, 'topic', $t_type == POST_GLOBAL ? $seo_core->seo_static['global_announce'] : $seo_core->seo_url['forum'][$f_id]);
						$t_url = append_sid("{$this->root_path}viewtopic.$this->php_ext", 'f=' . $f_id . '&amp;t=' . $t_id);
					}
					else
					{
						$t_url = append_sid("{$this->root_path}viewtopic.$this->php_ext", 'f=' . $f_id . '&amp;t=' . $t_id . "&amp;start=" . ($page * $per));
					}

					$t_url = $seo_core->drop_sid($t_url);
				}
				else
				{
					$f_part = (!isset($seo_topic_url) || (isset($seo_topic_url) && $seo_topic_url)) ? "" : "f=$f_id&amp;";

					if ($page == 0)
					{
						$t_url = $path . "viewtopic.$this->php_ext?" . $f_part . "t=$t_id";
					}
					else
					{
						$t_url = $path . "viewtopic.$this->php_ext?" . $f_part . "t=$t_id&amp;start=" . ($page * $per);
					}
				}

				if ($urls && ($urls % ($limit * $index)) == 0)
				{
					$f_xml = $f_beg . $f_xml . $f_end;

					$written = true;

					if (!($w = @fopen($this->root_path . $dir . $index . '.xml', 'w')))
					{
						$written = false;
					}

					if (!(@fwrite($w, $f_xml)))
					{
						$written = false;
					}

					@fclose($w);

					if (!$written)
					{
						trigger_error(sprintf($this->user->lang['SEOMAP_CANT_WRITE'], $path . $dir));
					}

					$f_xml = '';
					++$index;
				}

				$time = gmdate('Y-m-d\TH:i:s+00:00', (int) max($times));
				$freq = $this->config['sitemap_seo_freq_' . $t_type . ''];
				$prior = $this->config['sitemap_seo_prior_' . $t_type . ''];

				$this->add_to_record($t_url, $time, $freq, $prior);

				++$page;
				$times = array();
				$post = 0;

				if ($data['post_id'] == $data['topic_last_post_id'])
				{
					$page = 0;
				}
			}
		}
		$this->db->sql_freeresult($result);

		if ($f_xml)
		{
			$f_xml = $f_beg . $f_xml . $f_end;
		}
		else
		{
			trigger_error('SEOMAP_NO_DATA');
		}

		if ($index > 1)
		{
			if ($f_xml)
			{
				$written = true;

				if (!($w = @fopen($this->root_path . $dir . $index . '.xml', 'w')))
				{
					$written = false;
				}

				if (!(@fwrite($w, $f_xml)))
				{
					$written = false;
				}

				@fclose($w);

				if (!$written)
				{
					trigger_error(sprintf($this->user->lang['SEOMAP_CANT_WRITE'], $path . $dir));
				}
			}

			$f_xml = '<?xml version="1.0" encoding="UTF-8"?>'."\r\n";
			$f_xml .= '<?xml-stylesheet type="text/xsl" href="' . $path . 'ext/shredder/sitemap/styles/sitemap.xsl"?>'."\r\n";
			$f_xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">'."\r\n";

			for ($i = 1; $i <= $index; $i++)
			{
				$f_xml .= '<sitemap>'."\r\n";
				$f_xml .= '<loc>' . $path . $dir . $i . '.xml</loc>'."\r\n";
				$f_xml .= '<lastmod>' . gmdate('Y-m-d\TH:i:s+00:00', time()) . '</lastmod>'."\r\n";
				$f_xml .= '</sitemap>'."\r\n";
			}

			$f_xml .= '</sitemapindex>';
		}
	}

	/**
	* Add URL record to sitemap
	*/
	public function add_to_record($loc, $time, $freq, $prior)
	{
		global $f_xml, $urls;

		$f_xml .= '<url>'."\r\n";
		$f_xml .= '<loc>'.$loc.'</loc>'."\r\n";

		if ($time)
		{
			$f_xml .= '<lastmod>'.$time.'</lastmod>'."\r\n";
		}
		if ($freq)
		{
			$f_xml .= '<changefreq>'.$freq.'</changefreq>'."\r\n";
		}
		if ($prior)
		{
			$f_xml .= '<priority>'.$prior.'</priority>'."\r\n";
		}
		$f_xml .= '</url>'."\r\n";

		++$urls;
	}
}
